<?php

class Hostinger_Ai_Assistant_Content_Generation {

	private Hostinger_Ai_Assistant_Errors $error_handler;
	private Hostinger_Ai_Assistant_Helper $helper;

	function __construct() {
		$this->error_handler     = new Hostinger_Ai_Assistant_Errors();
		$this->helper            = new Hostinger_Ai_Assistant_Helper();
	}

	public function get_or_create_category( $category ): int {
		$category_id = category_exists( $category );
		if ( ! $category_id ) {
			$category_data = array(
				'cat_name'             => $category,
				'category_description' => '',
				'category_nicename'    => sanitize_title( $category )
			);
			$category_id   = wp_insert_category( $category_data );
		}

		return $category_id;
	}

	public function get_or_create_tags( string $tags ): array {
		$tags_array = explode( ',', $tags );
		$tags_ids   = array();
		foreach ( $tags_array as $tag_name ) {
			$tag = term_exists( $tag_name, 'post_tag' );
			if ( ! $tag ) {
				$tag_data = array(
					'slug' => sanitize_title( $tag_name ),
					'name' => $tag_name
				);
				$tag_id   = wp_insert_term( $tag_data['name'], 'post_tag', $tag_data );
				if ( is_wp_error( $tag_id ) ) {
					error_log( 'Failed tag create ' . print_r( $tag_id, true ) );
				}
				$tag_id = $tag_id['term_id'];
			} else {
				$tag_id = $tag['term_id'];
			}
			$tags_ids[] = (int) $tag_id;
		}

		return $tags_ids;
	}

	public function create_post( string $post_type, string $title, string $content, int $category_id, array $tags_ids, int $featured_image_id = 0, string $post_status = 'draft' ): int {
		$amplitude = new Hostinger_Ai_Assistant_Amplitude();
		$post_data = array(
			'post_title'    => $title,
			'post_content'  => $content,
			'post_status'   => $post_status,
			'post_category' => array( $category_id ),
			'tags_input'    => $tags_ids,
			'post_type'     => $post_type
		);

		$post_id   = wp_insert_post( $post_data );

		if( $post_status === 'publish' ) {
			$amplitude->ai_content_published( $post_type, $post_id );
		}
		// Save meta field when content generated by AI.
		update_post_meta( $post_id, 'hostinger_ai_generated', true );
		$amplitude->ai_content_saved( $post_type, $post_id );

		if ( $featured_image_id !== 0 ) {
			set_post_thumbnail( $post_id, $featured_image_id );
		}

		return $post_id;
	}

	public function upload_image_to_media_library( string $image_path, object $image_response_data ): int {

		$image_url              = $image_path;
		$existing_attachment_id = attachment_url_to_postid( $image_url );
		if ( $existing_attachment_id ) {
			return $existing_attachment_id;
		}

		$image_data = file_get_contents( $image_url );

		if ( ! $image_data ) {
			error_log( 'image_fetch_failed: ' . $image_url );
		}

		$file_name        = basename( $image_url );
		$unique_file_name = wp_unique_filename( wp_upload_dir()['path'], $file_name );
		$upload_file      = wp_upload_bits( $unique_file_name, null, $image_data );

		if ( ! $upload_file['error'] ) {
			$attachment = array(
				'guid'           => $upload_file['url'],
				'post_mime_type' => $upload_file['type'],
				'post_title'     => sanitize_file_name( str_replace( '.jpg', '', $file_name ) ),
				'post_content'   => sanitize_text_field( $image_response_data->alt_description ) ?? '',
				'post_status'    => 'inherit',
				'post_excerpt'   => $this->unsplash_user_caption( $image_response_data ) ?? '',
				'post_author'    => get_current_user_id(),
			);

			$attachment_id = wp_insert_attachment( $attachment, $upload_file['file'] );
			$this->generate_attachment_meta_data( $image_response_data, $attachment_id, $upload_file['file'] );

			if ( $attachment_id ) {
				unlink( $image_path );
			}

			return $attachment_id;
		} else {
			error_log( 'image_upload_failed: ' . $upload_file['error'] );
		}
	}

	private function generate_attachment_meta_data( object $image_response_data, int $attachment_id, string $upload_file ): void {
		require_once ABSPATH . 'wp-admin/includes/image.php';
		//Disable image sizes generation
		add_filter( 'intermediate_image_sizes', '__return_empty_array' );
		$attachment_data = wp_generate_attachment_metadata( $attachment_id, $upload_file );
		remove_filter( 'intermediate_image_sizes', '__return_empty_array' );

		if ( $attachment_data ) {
			$unsplash_author_data = $this->generate_unsplash_author_data( $image_response_data );
			$attachment_data      = array_merge( $attachment_data, $unsplash_author_data );
			wp_update_attachment_metadata( $attachment_id, $attachment_data );
		}
	}

	private function generate_unsplash_author_data( object $image_response_data ): array {

		$utmSource            = '?utm_source=hostinger&utm_medium=referral';
		$unsplash_author_data = array(
			'unsplash_author_name'    => $image_response_data->user->name ?? '',
			'unsplash_author_profile' => $image_response_data->user->profile_url . $utmSource ?? '',
			'unsplash_referral_link'  => 'https://unsplash.com/' . $utmSource ?? '',
		);

		return $unsplash_author_data;
	}

	public function unsplash_user_caption( object $image_response_data ): string {

		$utmSource = 'hostinger';

		if ( ! isset( $image_response_data->user ) || empty( $image_response_data->user ) ) {
			return '';
		}

		$user_url  = ( isset( $image_response_data->user->profile_url ) ) ? $image_response_data->user->profile_url : '';
		$user_name = ( isset( $image_response_data->user->name ) ) ? $image_response_data->user->name : '';

		if ( empty( $user_url ) || empty( $user_name ) ) {
			return '';
		}

		$referral_url = add_query_arg(
			[
				'utm_source' => $utmSource,
				'utm_medium' => 'referral',
			],
			'https://unsplash.com/'
		);

		$format_link   = '<a href="%1$s" rel="nofollow">%2$s</a>';
		$user_link     = sprintf(
			$format_link,
			esc_url( $user_url ),
			esc_html( $user_name )
		);
		$referral_link = sprintf(
			$format_link,
			esc_url( $referral_url ),
			esc_html__( 'Unsplash', 'hostinger-ai-assistant' )
		);

		return sprintf( esc_html__( 'Photo by %1$s on %2$s ', 'unsplash' ), $user_link, $referral_link );
	}

	public function get_public_post_types(): array {

		$post_types                        = get_post_types( array(
			'public' => true,
		) );
		$post_types_with_content_and_title = array();

		foreach ( $post_types as $post_type ) {

			if ( post_type_supports( $post_type, 'editor' ) && post_type_supports( $post_type, 'title' ) ) {
				$post_types_with_content_and_title[] = $post_type;
			}
		}

		return $post_types_with_content_and_title;
	}

	public function process_post_action( string $action ): void {
		$error_msg    = $this->error_handler->get_error_message( 'action_failed' );
		$server_error = $this->error_handler->get_error_message( 'server_error' );

		try {
			$request         = new Hostinger_Ai_Assistant_Requests();
			$seo             = new Hostinger_Ai_Assistant_Seo();
			$nonce           = isset( $_POST['nonce'] ) ? sanitize_text_field( $_POST['nonce'] ) : '';
			$title           = isset( $_POST['title'] ) ? sanitize_text_field( $_POST['title'] ) : '';
			$content         = isset( $_POST['content'] ) ? wp_kses_post( $this->helper->sanitize_html_string( $_POST['content'] ) ) : '';
			$category        = isset( $_POST['category'] ) ? sanitize_text_field( $_POST['category'] ) : '';
			$tags            = isset( $_POST['tags'] ) ? sanitize_text_field( $_POST['tags'] ) : '';
			$post_type       = isset( $_POST['post_type'] ) ? sanitize_text_field( $_POST['post_type'] ) : '';
			$seo_keywords    = isset( $_POST['seo_keywords'] ) ? sanitize_text_field( $_POST['seo_keywords'] ) : '';
			$seo_description = isset( $_POST['seo_description'] ) ? sanitize_text_field( $_POST['seo_description'] ) : '';
			$post_type       = $this->validate_post_type( $post_type );

			if ( ! wp_verify_nonce( $nonce, ( $action === 'create' ) ? 'create_post' : 'publish_post' ) ) {
				$this->helper->ajax_error_message( $error_msg, $error_msg );

				return;
			}

			if ( $this->helper->has_taxonomy_for_post_type( $post_type, 'category' ) ) {
				$category_id = $this->get_or_create_category( $category );
			}

			if ( $this->helper->has_taxonomy_for_post_type( $post_type, 'post_tag' ) ) {
				$tags_ids = $this->get_or_create_tags( $tags );
			}

			if ( $this->helper->post_type_supports_featured_image( $post_type ) ) {
				$image_data = $request->get_unsplash_image_data( $tags );
				if ( ! empty( get_object_vars( $image_data ) ) ) {
					$featured_image_id = $request->get_uploaded_unsplash_image( $image_data );
				}
			}

			$post_status  = ( $action === 'create' ) ? '' : 'publish';
			$post_id = $this->create_post(
				$post_type,
				$title,
				$content,
				$category_id ?? 0,
				$tags_ids ?? [],
				$featured_image_id ?? 0,
				$post_status
			);

			$seo->add_seo_meta_data( $seo_keywords, $seo_description, $post_id );

			if ( $post_id ) {
				if ( $action === 'create' ) {
					$editor_url = 'post.php?action=edit&post=' . $post_id;
					wp_send_json_success( $editor_url );
				} else {
					$post_link = get_permalink( $post_id );

					if ( $this->helper->is_preview_domain() ) {
						$formatted_host = $this->helper->format_preview_domain( parse_url( $post_link, PHP_URL_HOST ) );
						$post_link      = $this->helper->overwrite_url_host( $post_link, $formatted_host );
					}
					wp_send_json_success( $post_link );
				}

				return;
			}

			wp_send_json_error( $error_msg );
		} catch ( Exception $exception ) {
			$this->helper->ajax_error_message( 'Error: ' . $exception->getMessage(), $server_error );
		}
	}

	public function validate_post_type( $post_type ): string {
		$allowed_post_types = array( 'post', 'product', 'page' );

		return in_array( $post_type, $allowed_post_types ) ? $post_type : 'page';
	}

	public function map_post_type( string $post_type ): string {
		switch ( $post_type ) {
			case 'post':
				return 'blog_post';
			case 'product':
				return 'product_description';
			default:
				return $post_type;
		}
	}

	public function validate_content_length( $content_length ): string {
		$allowed_lengths = array(
			'short'  => array( 'min' => 150, 'max' => 300 ),
			'medium' => array( 'min' => 300, 'max' => 500 ),
			'long'   => array( 'min' => 500, 'max' => 700 ),
		);

		if ( isset( $allowed_lengths[ $content_length ] ) ) {
			$length_range = $allowed_lengths[ $content_length ];

			return $length_range['min'] . '-' . $length_range['max'];
		} else {
			return '150-300';
		}
	}

}
